// content.js — HUD V4 (API-first + "ตั้งรหัสใหม่" temporary page flow)
// ใช้ร่วมกับ Cloudflare Worker hud-api-v2.kingking090ai.workers.dev
//
// *** VERSION 2 (Patched by Gemini) ***
// - Fixed session state (uses localStorage instead of /activeMaster)
// - Fixed Delete flow (prompts for password)
// - Fixed Password Reset flow (opens correct GET URL)

(() => {
  const API = "https://hud-api-v2.kingking090ai.workers.dev";
  const LS_KEY = 'hud_activeMaster'; // Key for client-side session

  // ---- Helpers -------------------------------------------------------------
  async function fetchJSON(path, opts = {}) {
    const res = await fetch(API + path, {
      method: opts.method || "GET",
      headers: {
        "content-type": "application/json",
        ...(opts.headers || {}),
      },
      body: opts.body ? JSON.stringify(opts.body) : undefined,
      credentials: "omit",
      cache: "no-store",
      mode: "cors",
    });
    if (!res.ok) {
      const errData = await res.json().catch(() => ({}));
      throw new Error(errData?.error || `API ${opts.method || "GET"} ${path} -> ${res.status}`);
    }
    const ct = res.headers.get("content-type") || "";
    return ct.includes("application/json") ? res.json() : res.text();
  }

  // ---- API wrappers --------------------------------------------------------
  const api = {
    // *** REMOVED: getActiveMaster ***

    // *** CHANGED: to validateMaster, points to /master/validate ***
    validateMaster(name, password) {
      const body = { name };
      if (typeof password === "string") body.password = password;
      return fetchJSON("/master/validate", { method: "POST", body });
    },
    getMasters() {
      return fetchJSON("/masters");
    },
    createMaster(name, password) {
      return fetchJSON("/masters", { method: "POST", body: { name, password } });
    },
    // *** CHANGED: accepts password and sends it in body/header ***
    deleteMaster(name, password = "") {
      return fetchJSON(`/master/${encodeURIComponent(name)}`, {
        method: "DELETE",
        headers: { "x-master-pass": password }, // Also send as header
        body: { password } // Send in body
      });
    },
    // *** REMOVED: startPasswordReset (was pointing to wrong endpoint) ***
    
    getProfiles(master) {
      return fetchJSON(`/profiles/${encodeURIComponent(master)}`);
    },
    saveProfile(master, profile) {
      return fetchJSON(`/profiles/${encodeURIComponent(master)}/save`, {
        method: "POST",
        body: {
          id: profile.id,
          displayName: profile.displayName,
          username: profile.username ?? "",
          password: profile.password ?? "",
        },
      });
    },
  };

  // ---- Field Mapping -------------------------------------------------------
  const fieldMappings = {
    username:
      "#app > div > div > div > div.MdBox01 > div > form > fieldset > div:nth-child(2) > input[type=text]",
    password:
      "#app > div > div > div > div.MdBox01 > div > form > fieldset > div:nth-child(3) > input[type=password]",
  };

  function setInputValue(el, value) {
    if (!el) return;
    const proto =
      el instanceof HTMLInputElement
        ? HTMLInputElement.prototype
        : HTMLTextAreaElement.prototype;
    const setter = Object.getOwnPropertyDescriptor(proto, "value")?.set;
    if (setter) {
      setter.call(el, value);
    } else {
      el.value = value;
    }
    el.dispatchEvent(new Event("input", { bubbles: true }));
    el.dispatchEvent(new Event("change", { bubbles: true }));
    el.blur();
  }
  function fillData(profile) {
    try {
      const u = document.querySelector(fieldMappings.username);
      const p = document.querySelector(fieldMappings.password);
      if (u && typeof profile.username === "string") {
        setInputValue(u, profile.username);
        console.log("HUD: filled username");
      }
      if (p && typeof profile.password === "string") {
        setInputValue(p, profile.password);
        console.log("HUD: filled password");
      }
    } catch (e) {
      console.error("HUD fill error:", e);
    }
  }

  // ---- Minimal HUD UI (show on every page) --------------------------------
  function el(tag, attrs = {}, children = []) {
    const d = document.createElement(tag);
    Object.entries(attrs).forEach(([k, v]) => {
      if (k === "style" && typeof v === "object") Object.assign(d.style, v);
      else if (k.startsWith("on") && typeof v === "function") d[k] = v;
      else d.setAttribute(k, v);
    });
    (Array.isArray(children) ? children : [children]).forEach((c) => {
      if (c == null) return;
      d.appendChild(typeof c === "string" ? document.createTextNode(c) : c);
    });
    return d;
  }

  function hudStyles() {
    if (document.getElementById("hud-v4-style")) return;
    const s = el("style", { id: "hud-v4-style" });
    s.textContent = `
      #hud-v4 {
        position: fixed; z-index: 2147483647; right: 20px; top: 20px;
        width: 320px; min-height: 140px; background:#222; color:#e0e0e0;
        border:1px solid #444; border-radius:10px; box-shadow:0 8px 25px rgba(0,0,0,0.7);
        padding:12px; font-family: 'Segoe UI', Arial, sans-serif;
      }
      #hud-v4 h3 { margin: 0 0 8px; color:#2bb634; font-size:16px; }
      #hud-v4 .row { display:flex; gap:8px; margin-top:8px; }
      #hud-v4 button {
        background:#28a745; border:none; color:#fff; padding:8px 10px; border-radius:6px; cursor:pointer; font-weight:600;
      }
      #hud-v4 button.secondary { background:#6c757d; }
      #hud-v4 button.warning { background:#f0ad4e; }
      #hud-v4 button.danger { background:#dc3545; }
      #hud-v4 input, #hud-v4 select {
        background:#333; color:#fff; border:1px solid #555; border-radius:5px; padding:8px; width:100%;
      }
      #hud-v4 .list { max-height:200px; overflow:auto; border:1px solid #333; border-radius:6px; padding:6px; }
      #hud-v4 .item { display:flex; justify-content:space-between; align-items:center; background:#2e2e2e; border:1px solid #444; border-radius:6px; padding:8px; margin-bottom:6px; }
      #hud-choice {
        position: fixed; z-index: 2147483647; background:#2b2b2b; color:#fff; border:1px solid #444;
        border-radius:10px; padding:10px; box-shadow:0 10px 30px rgba(0,0,0,.5);
      }
      #hud-choice .row { display:flex; gap:8px; }
      #hud-choice button { flex: 1; }
    `;
    document.head.appendChild(s);
  }

  function openSelectMenu(anchorEl, name) {
    closeChoice();
    const rect = anchorEl.getBoundingClientRect();
    const pop = el("div", { id: "hud-choice", style: { left: `${rect.left}px`, top: `${rect.bottom + 6}px` } }, [
      el("div", { style: { marginBottom: "8px", fontWeight: "700" } }, `เลือก "${name}"`),
      el("div", { class: "row" }, [
        el("button", {
          class: "secondary",
          // *** CHANGED: Login flow ***
          onclick: async () => {
            const pwd = prompt(`ใส่รหัสของ "${name}"\n(ถ้าไม่มีรหัส ให้เว้นว่าง)`);
            if (pwd == null) return;
            try {
              // 1. Validate password with server
              await api.validateMaster(name, pwd);
              // 2. Save session to *client*
              localStorage.setItem(LS_KEY, name);
              // 3. Re-render
              await render();
            } catch (e) {
              alert("รหัสไม่ถูกต้อง หรือ API error: " + e.message);
            } finally {
              closeChoice();
            }
          },
        }, "ใส่รหัส"),
        el("button", {
          class: "warning",
          // *** CHANGED: Password Reset flow ***
          onclick: () => {
            // Just open the HTML page from the API
            window.open(`${API}/password-reset/start?name=${encodeURIComponent(name)}`, "_blank");
            closeChoice();
          },
        }, "ตั้งรหัสใหม่"),
      ]),
    ]);
    document.body.appendChild(pop);

    setTimeout(() => {
      document.addEventListener("mousedown", outsideCloser, true);
      window.addEventListener("blur", closeChoice, { once: true });
    }, 0);
  }
  function closeChoice() {
    const old = document.getElementById("hud-choice");
    if (old) old.remove();
    document.removeEventListener("mousedown", outsideCloser, true);
  }
  function outsideCloser(e) {
    const pop = document.getElementById("hud-choice");
    if (pop && !pop.contains(e.target)) closeChoice();
  }

  function showMasterChooser(state) {
    const { masters } = state;
    const root = el("div", { id: "hud-v4" }, [
      el("h3", {}, "เลือก Master Profile"),
      el("div", { class: "list", id: "hud-masters-list" }),
      el("div", { class: "row" }, [
        el("input", { id: "hud-master-input", placeholder: "ตั้งชื่อ Master Profile..." }),
        el("button", {
          onclick: async () => {
            const name = (document.getElementById("hud-master-input").value || "").trim();
            if (!name) return alert("กรุณากรอกชื่อ");
            const pwd = prompt(`ตั้งรหัสสำหรับ "${name}" (เว้นว่างได้)`);
            if (pwd === null) return; // User cancelled
            try {
              await api.createMaster(name, pwd || "");
              await render();
            } catch (e) {
              console.error("createMaster failed", e);
              alert("สร้างไม่สำเร็จ: " + e.message);
            }
          },
        }, "สร้าง"),
      ]),
      el("div", { class: "row" }, [
        el("button", { class: "secondary", onclick: render }, "รีเฟรช"),
      ]),
    ]);
    document.body.appendChild(root);

    const list = root.querySelector("#hud-masters-list");
    list.innerHTML = "";
    if (!Array.isArray(masters) || masters.length === 0) {
      list.appendChild(el("div", {}, "(ยังไม่มี Master)"));
    } else {
      masters.forEach((name) => {
        const btnSelect = el("button", {
          class: "warning",
          style: { marginRight: "6px" },
          onclick: (ev) => openSelectMenu(ev.currentTarget, name),
        }, "เลือก");
        const btnDel = el("button", {
          class: "danger",
          // *** CHANGED: Delete flow (prompts for password) ***
          onclick: async () => {
            if (!confirm(`ลบ "${name}" ทั้งชุด?`)) return;
            const pwd = prompt(`ใส่รหัสของ "${name}" เพื่อยืนยันการลบ\n(ถ้าไม่มีรหัส ให้เว้นว่าง)`);
            if (pwd === null) return; // User cancelled
            try {
              await api.deleteMaster(name, pwd || ""); // Pass password
              await render();
            } catch (e) {
              console.error("deleteMaster failed", e);
              alert("ลบไม่สำเร็จ (รหัสอาจไม่ถูกต้อง): " + e.message);
            }
          },
        }, "ลบ");

        const row = el("div", { class: "item" }, [ el("div", {}, name), el("div", {}, [btnSelect, btnDel]) ]);
        list.appendChild(row);
      });
    }
  }

  function showMainHud(state) {
    const { activeMaster, profiles } = state;
    const root = el("div", { id: "hud-v4" }, [
      el("h3", {}, `Master: [${activeMaster}]`),
      el("div", { class: "row" }, [
        el("button", {
          class: "secondary",
          // *** CHANGED: Logout flow (clears localStorage) ***
          onclick: async () => {
            try {
              localStorage.removeItem(LS_KEY);
              await render();
            } catch (e) {
              console.error("logout master failed", e);
            }
          },
        }, "สลับ Master"),
        el("button", { class: "secondary", onclick: render }, "รีเฟรช"),
      ]),
      el("div", { class: "list", id: "hud-profiles-list" }),
      el("div", { class: "row" }, [ el("input", { id: "hud-prof-name", placeholder: "ชื่อโปรไฟล์ (แสดงในลิสต์)" }) ]),
      el("div", { class: "row" }, [
        el("input", { id: "hud-prof-user", placeholder: "Username" }),
        el("input", { id: "hud-prof-pass", placeholder: "Password" }),
      ]),
      el("div", { class: "row" }, [
        el("button", {
          onclick: async () => {
            const displayName = document.getElementById("hud-prof-name").value.trim();
            const username = document.getElementById("hud-prof-user").value ?? "";
            const password = document.getElementById("hud-prof-pass").value ?? "";
            if (!displayName) return alert("กรอกชื่อโปรไฟล์ก่อน");
            const profile = { id: "profile_" + Date.now(), displayName, username, password };
            try {
              await api.saveProfile(activeMaster, profile);
              await render();
            } catch (e) {
              console.error("saveProfile failed", e);
              alert("บันทึกไม่สำเร็จ: " + e.message);
            }
          },
        }, "บันทึกโปรไฟล์"),
      ]),
    ]);
    document.body.appendChild(root);

    const list = root.querySelector("#hud-profiles-list");
    list.innerHTML = "";
    if (!Array.isArray(profiles) || profiles.length === 0) {
      list.appendChild(el("div", {}, "ยังไม่มีโปรไฟล์"));
    } else {
      profiles.forEach((p) => {
        const row = el("div", { class: "item" }, [
          el("div", {}, p.displayName || p.id),
          el("div", {}, [
            el("button", { class: "warning", style: { marginRight: "6px" }, onclick: () => fillData(p) }, "ยิงข้อมูล"),
          ]),
        ]);
        list.appendChild(row);
      });
    }
  }

  // ---- Render Flow ---------------------------------------------------------
  async function render() {
    try {
      hudStyles();
      const old = document.getElementById("hud-v4");
      if (old) old.remove();

      // *** CHANGED: Read session from localStorage, not API ***
      let activeMaster = localStorage.getItem(LS_KEY) || "";

      if (!activeMaster) {
        // Not logged in, show chooser
        let masters = [];
        try {
          const r = await api.getMasters();
          masters = Array.isArray(r?.masters) ? r.masters : r || [];
        } catch (e) {
          console.warn("getMasters failed", e);
        }
        showMasterChooser({ masters });
      } else {
        // Logged in, show main HUD
        let profiles = [];
        try {
          const r = await api.getProfiles(activeMaster);
          profiles = Array.isArray(r?.profiles) ? r.profiles : (Array.isArray(r) ? r : []);
        } catch (e) {
          console.warn("getProfiles failed", e);
          // *** ADDED: If token is bad (e.g. master deleted), log out
          if (e.message?.includes("404") || e.message?.includes("not found")) {
            console.warn("Master not found, logging out client.");
            localStorage.removeItem(LS_KEY);
            await render(); // Re-render (will show chooser)
            return;
          }
        }
        showMainHud({ activeMaster, profiles });
      }
    } catch (e) {
      console.error("HUD render error:", e);
    }
  }

  if (document.readyState === "loading") document.addEventListener("DOMContentLoaded", render);
  else render();
})();